function stdcell_array_gui
% STD-CELL ARRAY GUI — duplicate the structure along Y
% Cells #1..#(N-1) = CUT Y far (flush to Ti), cell #N = STANDARD. N >= 3.
% Y pitch = width (after cut) of the CUT cell (if applied), otherwise W.

%% ========= DEFAULT VALUES =========
P = struct( ...
    'L',24, 'W',10, 'H',2, ...              % SiO2 base
    'origin',[0 0 0], ...                   % [x0 y0 z0]
    'H_Ti_bottom',1, ...                    % Ti bottom (height only)
    'H_gold_bottom',2, ...                  % Au bottom (height only)
    'L_vac',4, 'H_vac',3, ...               % vacuum (Lx, Hz)
    'L_Au_top',16, 'W_Au_top',8, 'H_Au_top',4, ... % Au top (Lx, Wy, Hz)
    'T_film',0.5 ...                        % Ti film thickness
);
N_default = 3;                 % number of cells (>=3)
CLK_default = '3,0,-3';        % [Switch,Hold,Reset] V

%% ========= GUI INTERFACE =========
ui = uifigure('Name','STD-CELL Array (Y) — last standard, others cut', ...
              'Color','w','Position',[80 80 1180 700]);

gl = uigridlayout(ui,[2 2]);
gl.RowHeight   = {'1x', 46};
gl.ColumnWidth = {400, '1x'};

% --- Control panel
ctrl = uipanel(gl,'Title','Parametri','FontWeight','bold');
ctrl.Layout.Row = 1; ctrl.Layout.Column = 1;

cg = uigridlayout(ctrl,[21 2]);
cg.RowHeight   = repmat({28},1,21);
cg.ColumnWidth = {200, 180};

r=1;
ef_L   = addField_arrayY(cg, r, 'L  [nm]',         P.L,            [0 Inf]); r=r+1;
ef_W   = addField_arrayY(cg, r, 'W  [nm]',         P.W,            [0 Inf]); r=r+1;
ef_H   = addField_arrayY(cg, r, 'H  [nm]',         P.H,            [0 Inf]); r=r+1;

ef_Hti = addField_arrayY(cg, r, 'H_{Ti,bot} [nm]', P.H_Ti_bottom,  [0 Inf]); r=r+1;
ef_Hau = addField_arrayY(cg, r, 'H_{Au,bot} [nm]', P.H_gold_bottom,[0 Inf]); r=r+1;

ef_Lv  = addField_arrayY(cg, r, 'L_{vac} [nm]',    P.L_vac,        [0 Inf]); r=r+1;
ef_Hv  = addField_arrayY(cg, r, 'H_{vac} [nm]',    P.H_vac,        [0 Inf]); r=r+1;

ef_Lat = addField_arrayY(cg, r, 'L_{Au,top} [nm]', P.L_Au_top,     [0 Inf]); r=r+1;
ef_Wat = addField_arrayY(cg, r, 'W_{Au,top} [nm]', P.W_Au_top,     [0 Inf]); r=r+1;
ef_Hat = addField_arrayY(cg, r, 'H_{Au,top} [nm]', P.H_Au_top,     [0 Inf]); r=r+1;

ef_Tf  = addField_arrayY(cg, r, 'T_{film,Ti} [nm]',P.T_film,       [0 Inf]); r=r+1;

% Origin (=> set Layout *after* creation)
lbl = uilabel(cg,'Text','origin x0 [nm]','HorizontalAlignment','right','FontWeight','bold');
lbl.Layout.Row = r; lbl.Layout.Column = 1;
uix0 = uieditfield(cg,'numeric','Value',P.origin(1),'Limits',[-Inf Inf]);
uix0.Layout.Row = r; uix0.Layout.Column = 2; r=r+1;

lbl = uilabel(cg,'Text','origin y0 [nm]','HorizontalAlignment','right','FontWeight','bold');
lbl.Layout.Row = r; lbl.Layout.Column = 1;
uiy0 = uieditfield(cg,'numeric','Value',P.origin(2),'Limits',[-Inf Inf]);
uiy0.Layout.Row = r; uiy0.Layout.Column = 2; r=r+1;

lbl = uilabel(cg,'Text','origin z0 [nm]','HorizontalAlignment','right','FontWeight','bold');
lbl.Layout.Row = r; lbl.Layout.Column = 1;
uiz0 = uieditfield(cg,'numeric','Value',P.origin(3),'Limits',[-Inf Inf]);
uiz0.Layout.Row = r; uiz0.Layout.Column = 2; r=r+1;

% N cells (>=3)
lblN = uilabel(cg,'Text','N celle (>=3)','HorizontalAlignment','right','FontWeight','bold');
lblN.Layout.Row = r; lblN.Layout.Column = 1;
ef_N = uieditfield(cg,'numeric','Value',N_default,'Limits',[3 Inf],'RoundFractionalValues','on'); 
ef_N.Layout.Row = r; ef_N.Layout.Column = 2; r=r+1;

% Clock voltages
lblC = uilabel(cg,'Text','Clock [S,H,R] (V)','HorizontalAlignment','right','FontWeight','bold');
lblC.Layout.Row = r; lblC.Layout.Column = 1;
ef_clk = uieditfield(cg,'text','Value',CLK_default);
ef_clk.Layout.Row = r; ef_clk.Layout.Column = 2; r=r+1;

% --- Buttons
btnPanel = uipanel(gl,'BorderType','none'); btnPanel.Layout.Row=2; btnPanel.Layout.Column=[1 2];
bgl = uigridlayout(btnPanel,[1 7]); bgl.ColumnWidth = {'fit','fit','fit','fit','1x','fit','fit'};

uibutton(bgl,'Text','Reset default','ButtonPushedFcn',@(~,~)resetDefaults());
uibutton(bgl,'Text','Aggiorna disegno','BackgroundColor',[0.1 0.5 0.1],'FontColor','w','ButtonPushedFcn',@(~,~)updateAndDraw());
uibutton(bgl,'Text','Export SDE .txt','BackgroundColor',[0.10 0.35 0.70],'FontColor','w','ButtonPushedFcn',@(~,~)exportSDE_arrayY());
uibutton(bgl,'Text','Export Sdevice.txt','BackgroundColor',[0.80 0.35 0.10],'FontColor','w','ButtonPushedFcn',@(~,~)exportSDevice_arrayY());
uilabel(bgl,'Text',''); % spacer
uibutton(bgl,'Text','Salva PNG','ButtonPushedFcn',@(~,~)savePNG());
uibutton(bgl,'Text','Chiudi','ButtonPushedFcn',@(~,~)close(ui));

% --- Graphics area
axPanel = uipanel(gl,'Title','Anteprima array (Y)','FontWeight','bold'); 
axPanel.Layout.Row=1; axPanel.Layout.Column=2;
ax = uiaxes('Parent',axPanel,'BackgroundColor','w'); ax.Units = 'normalized'; ax.Position = [0 0 1 1];

% Auto-update
allEf = [ef_L,ef_W,ef_H,ef_Hti,ef_Hau,ef_Lv,ef_Hv,ef_Lat,ef_Wat,ef_Hat,ef_Tf,uix0,uiy0,uiz0,ef_N];
for k=1:numel(allEf)
    allEf(k).ValueChangedFcn = @(~,~)updateAndDraw();
end

% First draw
updateAndDraw();

%% ===== nested GUI functions =====
    function ef = addField_arrayY(parent, row, lbl, val, lim)
        hLbl = uilabel(parent,'Text',lbl,'HorizontalAlignment','right','FontWeight','bold');
        hLbl.Layout.Row = row; hLbl.Layout.Column = 1;
        ef = uieditfield(parent,'numeric','Value',val, ...
            'Limits',lim, 'LowerLimitInclusive','on','UpperLimitInclusive','on', ...
            'ValueDisplayFormat','%.6g');
        ef.Layout.Row = row; ef.Layout.Column = 2;
    end

    function resetDefaults()
        % Restore default parameter struct
        P = struct('L',24,'W',10,'H',2,'origin',[0 0 0], ...
                   'H_Ti_bottom',1,'H_gold_bottom',2, ...
                   'L_vac',4,'H_vac',3, ...
                   'L_Au_top',16,'W_Au_top',8,'H_Au_top',4, ...
                   'T_film',0.5);
        % Reset GUI numeric fields
        ef_L.Value=P.L; ef_W.Value=P.W; ef_H.Value=P.H;
        ef_Hti.Value=P.H_Ti_bottom; ef_Hau.Value=P.H_gold_bottom;
        ef_Lv.Value=P.L_vac; ef_Hv.Value=P.H_vac;
        ef_Lat.Value=P.L_Au_top; ef_Wat.Value=P.W_Au_top; ef_Hat.Value=P.H_Au_top;
        ef_Tf.Value=P.T_film;
        uix0.Value=P.origin(1); uiy0.Value=P.origin(2); uiz0.Value=P.origin(3);
        ef_N.Value = max(N_default,3);
        ef_clk.Value = CLK_default;
        updateAndDraw();
    end

    function updateAndDraw()
        % read from form
        P.L = ef_L.Value; P.W = ef_W.Value; P.H = ef_H.Value;
        P.H_Ti_bottom = ef_Hti.Value; P.H_gold_bottom = ef_Hau.Value;
        P.L_vac = ef_Lv.Value; P.H_vac = ef_Hv.Value;
        P.L_Au_top = ef_Lat.Value; P.W_Au_top = ef_Wat.Value; P.H_Au_top = ef_Hat.Value;
        P.T_film = ef_Tf.Value;
        P.origin = [uix0.Value, uiy0.Value, uiz0.Value];
        N = max(round(ef_N.Value),3);    % minimum 3
        ef_N.Value = N;

        % draw array
        drawArrayY(ax, P, N);
    end

    function savePNG()
        [f,p] = uiputfile('stdcell_array_y.png','Salva come');
        if isequal(f,0), return; end
        exportgraphics(ax, fullfile(p,f), 'Resolution',300);
    end

    % =========================  EXPORT SDE (ARRAY, N CELLS)  =========================
    function exportSDE_arrayY()
        % ----- read GUI parameters
        P.L = ef_L.Value; P.W = ef_W.Value; P.H = ef_H.Value;
        P.H_Ti_bottom = ef_Hti.Value; P.H_gold_bottom = ef_Hau.Value;
        P.L_vac = ef_Lv.Value; P.H_vac = ef_Hv.Value;
        P.L_Au_top = ef_Lat.Value; P.W_Au_top = ef_Wat.Value; P.H_Au_top = ef_Hat.Value;
        P.T_film = ef_Tf.Value;
        N = max(round(ef_N.Value),3);

        % ----- conversion nm -> µm
        nm2um = @(x) x*1e-3;
        BottomDielectricWidth     = nm2um(P.L);
        TrenchMiddleWidth         = nm2um(P.L_vac);
        BottomDielectricThickness = nm2um(P.H);
        CellLength                = nm2um(P.W);
        TrenchWallHeight          = nm2um(P.H_vac);
        AdhesionLayerThickness    = nm2um(P.H_Ti_bottom);
        GoldLayerThickness        = nm2um(P.H_gold_bottom);
        TopGoldWidthX             = nm2um(P.L_Au_top);
        TopGoldHeightY            = nm2um(P.W_Au_top);
        TopGoldThicknessZ         = nm2um(P.H_Au_top);
        TitaniumRimThickness      = nm2um(P.T_film);
        ViaThickness              = 0.001000; % µm
        PhasesNumber              = 1;

        % ----- text buffer
        buf = {}; 
        function push(fmt,varargin), buf{end+1} = sprintf(fmt,varargin{:}); end

        % ===== HEADER + PARAMETERS =====
        push(';; ============================================================\n');
        push(';; SENTARUS SDE — ARRAY Y with N=%d (cells 1-%d CUT, cell %d STANDARD)\n', N, N-1, N);
        push(';; SiO2 — Dielectric   HfO2 — Side bodies   Ti — Rim   Au/Ti — Contacts\n');
        push(';; Units: micrometers (µm)\n');
        push(';; ============================================================\n\n');
        push('(sde:clear)\n\n');

        push(';; -------------------------------\n');
        push(';; MAIN PARAMETERS (µm)\n');
        push(';; -------------------------------\n');
        push('(sde:define-parameter "BottomDielectricWidth"      %.6f)\n', BottomDielectricWidth);
        push('(sde:define-parameter "TrenchMiddleWidth"          %.6f)\n', TrenchMiddleWidth);
        push('(sde:define-parameter "TrenchWallWidth"            (/ (- BottomDielectricWidth TrenchMiddleWidth) 2))\n');
        push('(sde:define-parameter "BottomDielectricThickness"  %.6f)\n', BottomDielectricThickness);
        push('(sde:define-parameter "CellLength"                 %.6f)\n', CellLength);
        push('(sde:define-parameter "TrenchWallHeight"           %.6f)\n', TrenchWallHeight);
        push('(sde:define-parameter "AdhesionLayerThickness"     %.6f)\n', AdhesionLayerThickness);
        push('(sde:define-parameter "GoldLayerThickness"         %.6f)\n', GoldLayerThickness);
        push('(sde:define-parameter "ViaThickness"               %.6f)\n', ViaThickness);
        push('(sde:define-parameter "PhasesNumber"               %d)\n\n', PhasesNumber);

        push(';; -------------------------------\n');
        push(';; TOP GOLD + RIM (µm)\n');
        push(';; -------------------------------\n');
        push('(sde:define-parameter "TopGoldWidthX"        %.6f)\n', TopGoldWidthX);
        push('(sde:define-parameter "TopGoldHeightY"       %.6f)\n', TopGoldHeightY);
        push('(sde:define-parameter "TopGoldThicknessZ"    %.6f)\n', TopGoldThicknessZ);
        push('(sde:define-parameter "TitaniumRimThickness" %.6f)\n\n', TitaniumRimThickness);

        push(';; -------------------------------\n');
        push(';; SECONDARY PARAMETERS (global)\n');
        push(';; -------------------------------\n');
        push('(sde:define-parameter "x_2ndWallStart"            (+ TrenchWallWidth TrenchMiddleWidth))\n');
        push('(sde:define-parameter "z_BottomAdhesionLayerEnd"  (+ BottomDielectricThickness AdhesionLayerThickness))\n');
        push('(sde:define-parameter "z_MiddleGoldEnd"           (+ BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness))\n');
        push('(sde:define-parameter "z_WallEnd"                 (+ z_MiddleGoldEnd TrenchWallHeight))\n');
        push('(sde:define-parameter "z_2ndGoldLayerEnd"         (+ z_WallEnd TopGoldThicknessZ))\n');
        push('(sde:define-parameter "CenterX"                   (/ BottomDielectricWidth 2.0))\n');
        push('(sde:define-parameter "CenterY"                   (/ CellLength 2.0))\n\n');

        push(';; Bounding boxes (top Au and Ti rim) — local to the cell (0..CellLength)\n');
        push('(sde:define-parameter "x_Au_min"  (- CenterX (/ TopGoldWidthX  2.0)))\n');
        push('(sde:define-parameter "x_Au_max"  (+ CenterX (/ TopGoldWidthX  2.0)))\n');
        push('(sde:define-parameter "y_Au_min"  (- CenterY (/ TopGoldHeightY 2.0)))\n');
        push('(sde:define-parameter "y_Au_max"  (+ CenterY (/ TopGoldHeightY 2.0)))\n');
        push('(sde:define-parameter "x_Rim_min" (- x_Au_min TitaniumRimThickness))\n');
        push('(sde:define-parameter "x_Rim_max" (+ x_Au_max TitaniumRimThickness))\n');
        push('(sde:define-parameter "y_Rim_min" (- y_Au_min TitaniumRimThickness))\n');
        push('(sde:define-parameter "y_Rim_max" (+ y_Au_max TitaniumRimThickness))\n\n');

        push(';; -------------------------------\n');
        push(';; CELL-TO-CELL PITCH (if CUT, use cut width)\n');
        push(';; -------------------------------\n');
        push('(define RimPackWidth (+ TopGoldHeightY (* 2.0 TitaniumRimThickness)))\n');
        push('(define Pitch (if (> CellLength RimPackWidth)\n');
        push('                  (min CellLength (* 0.5 (+ CellLength TopGoldHeightY (* 2.0 TitaniumRimThickness))))\n');
        push('                  CellLength))\n\n');

        % y0_i offsets
        push(';; Offsets along Y for N=%d\n', N);
        for i=1:N
            if i==1, push('(define y0_%d 0.0)\n', i);
            else,     push('(define y0_%d (* %d.0 Pitch))\n', i, i-1);
            end
        end
        push('\n;; Helper functions for Y-shifted variables\n(define (y-shift local y0) (+ local y0))\n\n');

        % ============= CELLS =============
        for i=1:N
            isCut = (i~=N);
            push(';; ============================================================\n');
            push(';; ================   CELL #%d  (%s)   =================\n', i, tern_arrayY(isCut,'CUT Y far','STANDARD'));
            push(';; ============================================================\n');
            if isCut
                push('(define YMaxCut_%d (y-shift y_Rim_max y0_%d))        ;; cut flush with Ti rim\n', i,i);
            else
                push('(define YMaxCut_%d (+ y0_%d CellLength))     ;; no cut\n', i,i);
            end
            push('(define y_Au_min_%d (y-shift y_Au_min y0_%d))\n', i,i);
            push('(define y_Au_max_%d (y-shift y_Au_max y0_%d))\n', i,i);
            push('(define y_Rim_min_%d (y-shift y_Rim_min y0_%d))\n', i,i);
            push('(define y_Rim_max_%d (y-shift y_Rim_max y0_%d))\n\n', i,i);

            % LOWER STRUCTURE
            push(';; LOWER STRUCTURE\n');
            push('(sdegeo:create-cuboid (position 0        y0_%d    0)\n', i);
            push('                      (position BottomDielectricWidth YMaxCut_%d BottomDielectricThickness)\n', i);
            push('                      "SiO2")\n\n');
            push('(sdegeo:create-cuboid (position 0        y0_%d    BottomDielectricThickness)\n', i);
            push('                      (position BottomDielectricWidth YMaxCut_%d z_BottomAdhesionLayerEnd)\n', i);
            push('                      "Titanium")\n\n');
            push('(define BOTTOMCONTACT_%d\n', i);
            push('  (sdegeo:create-cuboid (position 0        y0_%d    z_BottomAdhesionLayerEnd)\n', i);
            push('                        (position BottomDielectricWidth YMaxCut_%d z_MiddleGoldEnd)\n', i);
            push('                        "Gold"))\n\n');

            % MIDDLE STRUCTURE
            push(';; MIDDLE STRUCTURE (Trench)\n');
            push('(sdegeo:create-cuboid (position TrenchWallWidth y0_%d z_MiddleGoldEnd)\n', i);
            push('                      (position x_2ndWallStart  YMaxCut_%d z_WallEnd)\n', i);
            push('                      "Vacuum")\n\n');
            push('(define BODY_LEFT_%d\n', i);
            push('  (sdegeo:create-cuboid (position 0               y0_%d    z_MiddleGoldEnd)\n', i);
            push('                        (position TrenchWallWidth YMaxCut_%d z_WallEnd)\n', i);
            push('                        "HfO2"))\n\n');
            push('(define BODY_RIGHT_%d\n', i);
            push('  (sdegeo:create-cuboid (position x_2ndWallStart  y0_%d    z_MiddleGoldEnd)\n', i);
            push('                        (position BottomDielectricWidth YMaxCut_%d z_WallEnd)\n', i);
            push('                        "HfO2"))\n\n');

            % UPPER LAYER (Top + Ti Rim)
            push(';; UPPER LAYER (Top Gold + Ti Rim)\n');
            push('(define TOPCONTACT_%d\n', i);
            push('  (sdegeo:create-cuboid (position x_Au_min y_Au_min_%d z_WallEnd)\n', i);
            push('                        (position x_Au_max y_Au_max_%d z_2ndGoldLayerEnd)\n', i);
            push('                        "Gold"))\n\n');
            push('(define RIM_LEFT_%d\n', i);
            push('  (sdegeo:create-cuboid (position x_Rim_min y_Rim_min_%d z_WallEnd)\n', i);
            push('                        (position x_Au_min  y_Rim_max_%d z_2ndGoldLayerEnd)\n', i);
            push('                        "Titanium"))\n');
            push('(define RIM_RIGHT_%d\n', i);
            push('  (sdegeo:create-cuboid (position x_Au_max  y_Rim_min_%d z_WallEnd)\n', i);
            push('                        (position x_Rim_max y_Rim_max_%d z_2ndGoldLayerEnd)\n', i);
            push('                        "Titanium"))\n');
            push('(define RIM_BOTTOM_%d\n', i);
            push('  (sdegeo:create-cuboid (position x_Rim_min y_Rim_min_%d z_WallEnd)\n', i);
            push('                        (position x_Rim_max y_Au_min_%d  z_2ndGoldLayerEnd)\n', i);
            push('                        "Titanium"))\n');
            push('(define RIM_TOP_%d\n', i);
            push('  (sdegeo:create-cuboid (position x_Rim_min y_Au_max_%d  z_WallEnd)\n', i);
            push('                        (position x_Rim_max y_Rim_max_%d z_2ndGoldLayerEnd)\n', i);
            push('                        "Titanium"))\n\n');
            push('(sdegeo:bool-unite (list RIM_LEFT_%d RIM_RIGHT_%d RIM_BOTTOM_%d RIM_TOP_%d))\n\n', i,i,i,i);

            % FILL
            if isCut
                push(';; SiO2 FILL around the rim (no FILL_TOP if zero thickness)\n');
                push('(define FILL_BOTTOM_%d\n', i);
                push('  (sdegeo:create-cuboid (position 0                 y0_%d        z_WallEnd)\n', i);
                push('                        (position BottomDielectricWidth y_Rim_min_%d z_2ndGoldLayerEnd)\n', i);
                push('                        "SiO2"))\n');
                push('(define FILL_LEFT_%d\n', i);
                push('  (sdegeo:create-cuboid (position 0         y_Rim_min_%d z_WallEnd)\n', i);
                push('                        (position x_Rim_min y_Rim_max_%d z_2ndGoldLayerEnd)\n', i);
                push('                        "SiO2"))\n');
                push('(define FILL_RIGHT_%d\n', i);
                push('  (sdegeo:create-cuboid (position x_Rim_max y_Rim_min_%d z_WallEnd)\n', i);
                push('                        (position BottomDielectricWidth y_Rim_max_%d z_2ndGoldLayerEnd)\n', i);
                push('                        "SiO2"))\n\n');
                push('(sdegeo:bool-unite (list FILL_BOTTOM_%d FILL_LEFT_%d FILL_RIGHT_%d))\n\n', i,i,i);
            else
                push(';; SiO2 FILL around the rim (STANDARD with TOP)\n');
                push('(define FILL_BOTTOM_%d\n', i);
                push('  (sdegeo:create-cuboid (position 0                 y0_%d        z_WallEnd)\n', i);
                push('                        (position BottomDielectricWidth y_Rim_min_%d z_2ndGoldLayerEnd)\n', i);
                push('                        "SiO2"))\n');
                push('(define FILL_LEFT_%d\n', i);
                push('  (sdegeo:create-cuboid (position 0         y_Rim_min_%d z_WallEnd)\n', i);
                push('                        (position x_Rim_min y_Rim_max_%d z_2ndGoldLayerEnd)\n', i);
                push('                        "SiO2"))\n');
                push('(define FILL_RIGHT_%d\n', i);
                push('  (sdegeo:create-cuboid (position x_Rim_max y_Rim_min_%d z_WallEnd)\n', i);
                push('                        (position BottomDielectricWidth y_Rim_max_%d z_2ndGoldLayerEnd)\n', i);
                push('                        "SiO2"))\n');
                push('(define FILL_TOP_%d\n', i);
                push('  (sdegeo:create-cuboid (position 0                 y_Rim_max_%d z_WallEnd)\n', i);
                push('                        (position BottomDielectricWidth YMaxCut_%d z_2ndGoldLayerEnd)\n', i);
                push('                        "SiO2"))\n\n');
                push('(sdegeo:bool-unite (list FILL_BOTTOM_%d FILL_LEFT_%d FILL_RIGHT_%d FILL_TOP_%d))\n\n', i,i,i,i);
            end
        end

        % ===== CONTACTS =====
        push(';; ============================================================\n');
        push(';; CONTACTS (one set for each top contact)\n');
        push(';; ============================================================\n');
        push('(sdegeo:define-contact-set "BottomContact" 4 (color:rgb 1 0 0) "##")\n');
        push('(sdegeo:set-current-contact-set "BottomContact")\n');
        for i=1:N, push('(sdegeo:set-contact BOTTOMCONTACT_%d "BottomContact")\n', i); end

        for i=1:N
            push('(sdegeo:define-contact-set "TopContact_%d" 4 (color:rgb 0 0 1) "##")\n', i);
            push('(sdegeo:set-current-contact-set "TopContact_%d")\n', i);
            push('(sdegeo:set-contact TOPCONTACT_%d "TopContact_%d")\n', i, i);
        end
        push('\n');

        % ===== MESH / REFINEMENT =====
        push(';; ============================================================\n');
        push(';; MESH / REFINEMENT (each cell within its window)\n');
        push(';; ============================================================\n');
        for i=1:N
            push(['(sdedr:define-refeval-window "InnerTrenchMesh_' num2str(i) '" "Cuboid"\n' ...
                  '  (position TrenchWallWidth y0_' num2str(i) ' z_BottomAdhesionLayerEnd)\n' ...
                  '  (position x_2ndWallStart  YMaxCut_' num2str(i) ' z_2ndGoldLayerEnd))\n\n']);
        end
        push('(sdedr:define-refinement-size "TrenchRefinement" 0.001 0.005 0.001 0.001 0.005 0.001)\n');
        for i=1:N
            push('(sdedr:define-refinement-placement "Refine_%d" "TrenchRefinement" (list "window" "InnerTrenchMesh_%d"))\n', i, i);
        end
        push('\n(sdedr:define-refinement-function "TrenchRefinement" "MaxLenInt" "Gold"    "Vacuum"   0.0001 "DoubleSide")\n');
        push('(sdedr:define-refinement-function "TrenchRefinement" "MaxLenInt" "Gold"    "Titanium" 0.0001)\n\n');

        % ===== BUILD =====
        push(';; ============================================================\n');
        push(';; BUILD + SAVE STRUCTURE\n');
        push(';; ============================================================\n');
        push('(sde:build-mesh "n@node@")\n');

        % ----- save file
        S = [buf{:}];
        [f,p] = uiputfile(sprintf('stdcell_arrayY_N%d_SDE.txt',N),'Export SDE .txt');
        if isequal(f,0), return; end
        fid = fopen(fullfile(p,f),'w');
        if fid < 0
            uialert(ui,'Impossibile creare il file di output.','Errore','Icon','error'); return;
        end
        fwrite(fid, S, 'char'); fclose(fid);
        uialert(ui, sprintf('File generato:\n%s', fullfile(p,f)), 'Export completato', 'Icon', 'success');
    end

    % ===================== EXPORT SDEVICE (ARRAY, multi-top) =====================
    function exportSDevice_arrayY()
        % Number of cells and phase voltages
        N = max(round(ef_N.Value),3);
        % parse clock "a,b,c"
        clk_txt = strrep(ef_clk.Value,' ','');
        vals = regexp(clk_txt,'^\s*([+-]?\d*\.?\d+(?:[eE][+-]?\d+)?)\s*,\s*([+-]?\d*\.?\d+(?:[eE][+-]?\d+)?)\s*,\s*([+-]?\d*\.?\d+(?:[eE][+-]?\d+)?)\s*$','tokens','once');
        if isempty(vals)
            Vswitch=3; Vhold=0; Vreset=-3;
        else
            Vswitch = str2double(vals{1});
            Vhold   = str2double(vals{2});
            Vreset  = str2double(vals{3});
        end

        % base phase for each contact: 0=S,1=H,2=R
        basePhase = arrayfun(@(i) mod(i-1,3), 1:N);

        % compute voltage vector for configuration k (1..3)
        function V = voltages_for_config(k)
            V = zeros(1,N);
            for ii=1:N
                ph = mod(basePhase(ii) + (k-1), 3);
                switch ph
                    case 0, V(ii)=Vswitch;
                    case 1, V(ii)=Vhold;
                    case 2, V(ii)=Vreset;
                end
            end
        end

        % Accumulate lines in cell array L
        L = {};
        function ap(s), L{end+1} = s; end

        % ---- Header (file block)
        ap('File {');
        ap('  Grid      = "n1_msh.tdr"');
        ap('  Plot      = "n@node@_clock_des.tdr"');
        ap('  Current   = "n@node@_clock_des.plt"');
        ap('  Parameter = "sdevice.par"');
        ap('}');
        ap('');

        % ---- Electrode block
        ap('Electrode {');
        ap('  { Name = "BottomContact"  Voltage = 0.0 }');
        for i=1:N
            ap(sprintf('  { Name = "TopContact_%d"   Voltage = 0.0 }', i));
        end
        ap('}');
        ap('');

        % ---- Physics & Thermode
        ap('# ----------------------------');
        ap('# PHYSICS');
        ap('# ----------------------------');
        ap('Physics (Material="HfO2") {');
        ap('  CondInsulator');
        ap('}');
        ap('');
        ap('Physics (Material="SiO2") {');
        ap('  CondInsulator');
        ap('}');
        ap('');
        ap('Thermode {');
        ap('  { Name = "BottomContact" Temperature = 300 SurfaceResistance = 1e-5 }');
        for i=1:N
            ap(sprintf('  { Name = "TopContact_%d" Temperature = 300 SurfaceResistance = 1e-5 }', i));
        end
        ap('}');
        ap('');

        % ---- Plot
        ap('# ----------------------------');
        ap('# PLOT');
        ap('# ----------------------------');
        ap('Plot {');
        ap('  Potential');
        ap('  ElectricField');
        ap('  DielectricConstant');
        ap('  Temperature');
        ap('  ConductionCurrentDensity/Vector');
        ap('  DisplacementCurrentDensity/Vector');
        ap('  TotalCurrent/Vector');
        ap('  SpaceCharge');
        ap('  Potential Doping');
        ap('  BandGap ElectronAffinity');
        ap('  ConductionBandEnergy ValenceBandEnergy');
        ap('}');
        ap('');
        ap('Math {');
        ap('  RelErrControl');
        ap('  Extrapolate');
        ap('}');
        ap('');

        % ---- Solve: equilibrium + 3 configurations (S/H/R rotate) with return-to-zero
        ap('Solve {');
        ap('  Coupled (Iterations= 100 LineSearchDamping= 1e-8) {Poisson}');
        ap('  Coupled{ Poisson Temperature Contact CondInsulator }');
        ap('  Plot(FilePrefix="n@node@_equilibrium")');
        ap('');

        for kCfg = 1:3
            Vcfg = voltages_for_config(kCfg);

            ap('  # =====================================================================');
            ap(sprintf('  # ==== CONFIG %d (phase rotation S/H/R) ====', kCfg));
            ap('  quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4');
            for i=1:N
                ap(sprintf('    Goal { name= "TopContact_%d" voltage = %g }', i, Vcfg(i)));
            end
            ap('    plot { range=(0, 1) intervals= 1}');
            ap('  ){coupled { Poisson Temperature CondInsulator }}');
            ap('');
            ap(sprintf('  Plot(FilePrefix="n@node@_cfg%d")', kCfg));
            ap('');

            ap('  # ---- return to 0 V on all top contacts');
            ap('  quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4');
            for i=1:N
                ap(sprintf('    Goal { name= "TopContact_%d" voltage = 0.0 }', i));
            end
            ap('    plot { range=(0, 1) intervals= 1}');
            ap('  ){coupled { Poisson Temperature CondInsulator }}');
            ap('');
        end

        ap('}');

        decktxt = strjoin(L, newline);

        [f,p] = uiputfile(sprintf('Sdevice_arrayY_mv_N%d.txt',N),'Export Sdevice.txt');
        if isequal(f,0), return; end

        fid = fopen(fullfile(p,f),'w');
        if fid<0
            uialert(ui,'Impossibile creare il file di output.','Errore','Icon','error');
            return;
        end
        fprintf(fid, '%s', decktxt);
        fclose(fid);
        uialert(ui, sprintf('File generato:\n%s', fullfile(p,f)), 'Export completato', 'Icon', 'success');
    end
end  % ====== END main function ======


% ======================================================================
% ===================== SUBFUNCTIONS (local scope) =====================
% ======================================================================

function drawArrayY(ax, P, N)
% Colors and style
[faceColor_base,faceColor_Ti,faceColor_Au,faceColor_vac,alpha_vac,faceColor_hfo2,edgeColor,lineWidth] = colorsAndStyle_arrayY();

% FULL axes reset
if ~isgraphics(ax), ax = gca; end
cla(ax,'reset');                 
delete(findall(ax,'Type','legend'));  
hold(ax,'on');

% Pitch = Y width of the CUT cell (if applied), otherwise W
pitch = calcPitchCutY_arrayY(P);

% Array along +Y, step = computed pitch
x0 = P.origin(1); y0 = P.origin(2); z0 = P.origin(3);

legendAdded = false;
for i = 1:N
    o_i = [x0, y0 + (i-1)*pitch, z0];

    % Only the last one (#N) is STANDARD; all others are CUT Y far
    isCut = (i ~= N);

    showLegend = ~legendAdded;  % legend only from the first cell
    drawOneCell_arrayY(ax, P, o_i, isCut, showLegend, ...
        faceColor_base,faceColor_Ti,faceColor_Au,faceColor_vac,alpha_vac,faceColor_hfo2,edgeColor,lineWidth);

    if showLegend, legendAdded = true; end
end

% Axes & views
axis(ax,'equal'); grid(ax,'on'); view(ax,35,25);
xlabel(ax,'X [nm]'); ylabel(ax,'Y [nm]'); zlabel(ax,'Z [nm]');
title(ax, sprintf('Standard Cell Array (Y) - N=%d, pitch=%.3g nm (last standard, others cut)', N, pitch));

% convenient limits
H_tot = P.H + P.H_Ti_bottom + P.H_gold_bottom + P.H_vac + P.H_Au_top;
Y_span = (N-1)*pitch + P.W;   
marg = 0.1 * [P.L max(P.W,Y_span) H_tot];
xlim(ax,[x0 - marg(1), x0 + P.L + marg(1)]);
ylim(ax,[y0 - marg(2), y0 + Y_span + marg(2)]);
zlim(ax,[z0 - marg(3), z0 + H_tot + marg(3)]);

plot3(ax,x0,y0,z0,'ko','MarkerFaceColor','k','HandleVisibility','off');
hold(ax,'off');
drawnow limitrate;
end

function pitch = calcPitchCutY_arrayY(P)
% Compute the inter-cell pitch considering the Ti rim pack
W_pack_ext = P.W_Au_top + 2*P.T_film;
if P.W > W_pack_ext
    pitch = min(P.W, 0.5*(P.W + P.W_Au_top + 2*P.T_film));
else
    pitch = P.W;  
end
end

function drawOneCell_arrayY(ax, P, o, isCut, showLegend, ...
        faceColor_base,faceColor_Ti,faceColor_Au,faceColor_vac,alpha_vac,faceColor_hfo2,edgeColor,lineWidth)

F = faces6_arrayY();

L=P.L; W=P.W; H=P.H;
H_Ti_bottom=P.H_Ti_bottom; H_gold_bottom=P.H_gold_bottom;
L_vac=P.L_vac; H_vac=P.H_vac;
L_Au_top=P.L_Au_top; W_Au_top=P.W_Au_top; H_Au_top=P.H_Au_top;
T_film=P.T_film;

% Base and bottom metal stack
V_base = vbox_arrayY(o, L, W, H);
o_Ti   = o + [0 0 H];
V_Ti   = vbox_arrayY(o_Ti, L, W, H_Ti_bottom);
o_Au   = o + [0 0 (H + H_Ti_bottom)];
V_Au   = vbox_arrayY(o_Au, L, W, H_gold_bottom);

% Middle trench (vacuum + HfO2 bodies)
z0_top = o(3) + H + H_Ti_bottom + H_gold_bottom;
L_vac_eff = min(max(L_vac,0), L);
L_left  = max(0, (L - L_vac_eff)/2);
L_right = max(0, L - (L_left + L_vac_eff));

o_left  = [o(1),                 o(2), z0_top];
o_vac   = [o(1) + L_left,        o(2), z0_top];
o_right = [o(1) + L_left + L_vac_eff, o(2), z0_top];

V_left  = vbox_arrayY(o_left,  L_left,    W, H_vac);
V_vac   = vbox_arrayY(o_vac,   L_vac_eff, W, H_vac);
V_right = vbox_arrayY(o_right, L_right,   W, H_vac);

% Top gold cap
x0_cap = o(1) + (L - L_Au_top)/2;
y0_cap = o(2) + (W - W_Au_top)/2;
z0_cap = z0_top + H_vac;
o_cap  = [x0_cap, y0_cap, z0_cap];
V_cap  = vbox_arrayY(o_cap, L_Au_top, W_Au_top, H_Au_top);

% Ti film (rim) around the cap
o_film_xL = [x0_cap - T_film,      y0_cap,              z0_cap];
o_film_xR = [x0_cap + L_Au_top,    y0_cap,              z0_cap];
V_film_xL = vbox_arrayY(o_film_xL, T_film,             W_Au_top, H_Au_top);
V_film_xR = vbox_arrayY(o_film_xR, T_film,             W_Au_top, H_Au_top);
o_film_yF = [x0_cap - T_film,      y0_cap - T_film,     z0_cap];
o_film_yB = [x0_cap - T_film,      y0_cap + W_Au_top,   z0_cap];
V_film_yF = vbox_arrayY(o_film_yF, L_Au_top + 2*T_film, T_film,   H_Au_top);
V_film_yB = vbox_arrayY(o_film_yB, L_Au_top + 2*T_film, T_film,   H_Au_top);

% External frame thicknesses (SiO2 around the rim pack)
x_in_min = x0_cap - T_film;
x_in_max = x0_cap + L_Au_top + T_film;
y_in_min = y0_cap - T_film;
y_in_max = y0_cap + W_Au_top + T_film;

x_base_min = o(1); x_base_max = o(1) + L;
y_base_min = o(2); y_base_max = o(2) + W;

T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0, y_base_max - y_in_max);

z0_frame = z0_cap; H_frame = H_Au_top;

V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
if T_left_frame  > 0, V_frame_xL = vbox_arrayY([x_base_min, y_base_min, z0_frame], T_left_frame,  W, H_frame); end
if T_right_frame > 0, V_frame_xR = vbox_arrayY([x_in_max,   y_base_min, z0_frame], T_right_frame, W, H_frame); end
if T_front_frame > 0, V_frame_yF = vbox_arrayY([x_in_min,   y_base_min, z0_frame], x_in_max - x_in_min, T_front_frame, H_frame); end
if T_back_frame  > 0, V_frame_yB = vbox_arrayY([x_in_min,   y_in_max,   z0_frame], x_in_max - x_in_min, T_back_frame,  H_frame); end

% Optional CUT on Y+
if isCut
    W_pack_ext = W_Au_top + 2*T_film;
    if W > W_pack_ext
        yMinCut = o(2);
        yMaxCut = min(y0_cap + W_Au_top + T_film, o(2) + W);
    else
        yMinCut = o(2);
        yMaxCut = o(2) + W;
    end

    V_base    = cropYBox_arrayY(V_base,    yMinCut,yMaxCut);
    V_Ti      = cropYBox_arrayY(V_Ti,      yMinCut,yMaxCut);
    V_Au      = cropYBox_arrayY(V_Au,      yMinCut,yMaxCut);
    V_left    = cropYBox_arrayY(V_left,    yMinCut,yMaxCut);
    V_vac     = cropYBox_arrayY(V_vac,     yMinCut,yMaxCut);
    V_right   = cropYBox_arrayY(V_right,   yMinCut,yMaxCut);
    V_cap     = cropYBox_arrayY(V_cap,     yMinCut,yMaxCut);
    V_film_xL = cropYBox_arrayY(V_film_xL, yMinCut,yMaxCut);
    V_film_xR = cropYBox_arrayY(V_film_xR, yMinCut,yMaxCut);
    V_film_yF = cropYBox_arrayY(V_film_yF, yMinCut,yMaxCut);
    V_film_yB = cropYBox_arrayY(V_film_yB, yMinCut,yMaxCut);

    x_base_min = o(1); x_base_max = o(1)+L;
    y_base_min = yMinCut; y_base_max = yMaxCut;

    T_left_frame  = max(0, x_in_min - x_base_min);
    T_right_frame = max(0, x_base_max - x_in_max);
    T_front_frame = max(0, y_in_min - y_base_min);
    T_back_frame  = max(0, y_base_max - y_in_max);

    V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
    if T_left_frame  > 0, V_frame_xL = vbox_arrayY([x_base_min, y_base_min, z0_frame], T_left_frame,  y_base_max-y_base_min, H_frame); end
    if T_right_frame > 0, V_frame_xR = vbox_arrayY([x_in_max,   y_base_min, z0_frame], T_right_frame, y_base_max-y_base_min, H_frame); end
    if T_front_frame > 0, V_frame_yF = vbox_arrayY([x_in_min,   y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame); end
    if T_back_frame  > 0, V_frame_yB = vbox_arrayY([x_in_min,   y_in_max,   z0_frame], max(0,x_in_max-x_in_min), T_back_frame,  H_frame); end
end

% Draw patches
drawPatch_arrayY(ax,V_base, faceColor_base, edgeColor,lineWidth, showLegend,'SiO2', 1, F);
drawPatch_arrayY(ax,V_Ti,   faceColor_Ti,   edgeColor,lineWidth, showLegend,'Ti',   1, F);
drawPatch_arrayY(ax,V_Au,   faceColor_Au,   edgeColor,lineWidth, showLegend,'Au',   1, F);

drawPatch_arrayY(ax,V_left, faceColor_hfo2, edgeColor,lineWidth, showLegend,'HfO2', 1, F);
drawPatch_arrayY(ax,V_vac,  faceColor_vac,  edgeColor,lineWidth, showLegend,'vacuum', 0.35, F);
drawPatch_arrayY(ax,V_right,faceColor_hfo2, edgeColor,lineWidth, false,'', 1, F, true); 

drawPatch_arrayY(ax,V_cap,     faceColor_Au, edgeColor,lineWidth, false,'', 1, F, true);
drawPatch_arrayY(ax,V_film_xL, faceColor_Ti, edgeColor,lineWidth, false,'', 1, F, true);
drawPatch_arrayY(ax,V_film_xR, faceColor_Ti, edgeColor,lineWidth, false,'', 1, F, true);
drawPatch_arrayY(ax,V_film_yF, faceColor_Ti, edgeColor,lineWidth, false,'', 1, F, true);
drawPatch_arrayY(ax,V_film_yB, faceColor_Ti, edgeColor,lineWidth, false,'', 1, F, true);

drawPatch_arrayY(ax,V_frame_xL, faceColor_base, edgeColor,lineWidth, false,'', 1, F, true);
drawPatch_arrayY(ax,V_frame_xR, faceColor_base, edgeColor,lineWidth, false,'', 1, F, true);
drawPatch_arrayY(ax,V_frame_yF, faceColor_base, edgeColor,lineWidth, false,'', 1, F, true);
drawPatch_arrayY(ax,V_frame_yB, faceColor_base, edgeColor,lineWidth, false,'', 1, F, true);
end

function Vc = cropYBox_arrayY(V, yMin, yMax)
% Crop a box in Y to the [yMin, yMax] interval
if isempty(V), Vc = []; return; end
oB  = V(1,:); LxB = V(2,1)-V(1,1); WyB = V(3,2)-V(1,2); HzB = V(4,3)-V(1,3);
Vc  = cropY_oLWH_arrayY(oB, LxB, WyB, HzB, yMin, yMax);
end

function Vc = cropY_oLWH_arrayY(oBox, Lx, Wy, Hz, yMin, yMax)
% Helper for Y-cropping from origin+LWH
y1 = oBox(2); y2 = oBox(2)+Wy;
if y2 <= yMin || y1 >= yMax, Vc = []; return; end
ya = max(y1, yMin); yb = min(y2, yMax); Wc = yb - ya;
if Wc <= 0, Vc = []; else, Vc = vbox_arrayY([oBox(1) ya oBox(3)], Lx, Wc, Hz); end
end

function V = vbox_arrayY(o,Lx,Wy,Hz)
% Return 8 vertices of an axis-aligned box at origin o with size Lx,Wy,Hz
V = [ o;
      o + [Lx 0 0];
      o + [0 Wy 0];
      o + [0 0 Hz];
      o + [Lx Wy 0];
      o + [Lx 0 Hz];
      o + [0 Wy Hz];
      o + [Lx Wy Hz] ];
end

function F = faces6_arrayY()
% 6 faces as quads over 8-vertex box
F = [1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
end

function [c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeColor,lineWidth] = colorsAndStyle_arrayY()
% Color palette and line styles
c_base  = [0.55 0.35 0.20]; % SiO2
c_ti    = [0.20 0.20 0.20]; % Ti
c_au    = [1.00 0.84 0.00]; % Au
c_vac   = [0.00 0.60 0.00]; % vacuum (visual aid)
a_vac   = 0.35;
c_hfo2  = [0.80 0.65 0.45]; % HfO2
edgeColor = 'k'; lineWidth = 1.0;
end

function ph = drawPatch_arrayY(axh,V,col,edgeCol,lw, addToLegend, name, faceAlpha, F, offLegend)
% Draw a patch with optional legend entry
if nargin<10, offLegend=false; end
if isempty(V), ph = []; return; end
if addToLegend && ~offLegend && ~isempty(name)
    ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
               'EdgeColor',edgeCol,'LineWidth',lw,'DisplayName',name);
else
    ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
               'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
end
end

function s = tern_arrayY(cond,a,b)
% Ternary helper
if cond, s=a; else, s=b; end
end
